/*
 * Decompiled with CFR 0.152.
 */
package org.syncfix.optimization;

import java.util.ArrayList;
import java.util.Queue;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.Mob;
import net.minecraft.world.entity.ai.Brain;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.phys.Vec3;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import org.syncfix.SyncFixMod;
import org.syncfix.optimization.EntityData;

public class AITickOptimizer {
    private static final int NEURAL_NETWORK_LAYERS = 3;
    private static final int NEURONS_PER_LAYER = 16;
    private static final double LEARNING_RATE = 0.001;
    private static final int PREDICTION_HORIZON = 10;
    private static final double OPTIMIZATION_THRESHOLD = 0.75;
    private static final int MAX_CONCURRENT_PREDICTIONS = 8;
    private final ConcurrentHashMap<UUID, AIEntityProfile> entityProfiles = new ConcurrentHashMap();
    private final ThreadPoolExecutor aiExecutor = new ThreadPoolExecutor(2, Math.max(4, Runtime.getRuntime().availableProcessors() / 2), 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(1000), r -> {
        Thread t = new Thread(r, "SyncFix-AI-" + System.nanoTime());
        t.setDaemon(true);
        t.setPriority(4);
        return t;
    });
    private final AtomicLong totalPredictions = new AtomicLong(0L);
    private final AtomicLong correctPredictions = new AtomicLong(0L);
    private final AtomicReference<Double> adaptiveThreshold = new AtomicReference<Double>(0.75);
    private final AtomicInteger activeNeuralNetworks = new AtomicInteger(0);
    private volatile boolean emergencyMode = false;
    private volatile long lastPerformanceCheck = 0L;

    public void initialize() {
        this.aiExecutor.prestartAllCoreThreads();
        SyncFixMod.LOGGER.info("Revolutionary AI Tick Optimizer initialized with {} neural networks", (Object)this.aiExecutor.getCorePoolSize());
    }

    public CompletableFuture<Boolean> shouldSkipTickAsync(Entity entity, EntityData data) {
        if (entity == null || entity.m_9236_().f_46443_ || !(entity instanceof LivingEntity) || entity instanceof Player || !entity.m_20197_().isEmpty()) {
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.supplyAsync(() -> {
            try {
                AIEntityProfile profile = this.entityProfiles.computeIfAbsent(entity.m_20148_(), k -> new AIEntityProfile(entity));
                return profile.predictOptimization(entity, data);
            }
            catch (Exception e) {
                return false;
            }
        }, this.aiExecutor).exceptionally(throwable -> false);
    }

    public boolean shouldSkipTick(Entity entity, EntityData data) {
        if (entity == null || entity.m_9236_().f_46443_ || !(entity instanceof LivingEntity) || entity instanceof Player || !entity.m_20197_().isEmpty()) {
            return false;
        }
        AIEntityProfile profile = this.entityProfiles.computeIfAbsent(entity.m_20148_(), k -> new AIEntityProfile(entity));
        return profile.predictOptimization(entity, data);
    }

    @SubscribeEvent
    public void onServerTick(TickEvent.ServerTickEvent event) {
        if (event.phase != TickEvent.Phase.END) {
            return;
        }
        MinecraftServer server = event.getServer();
        if (server == null) {
            return;
        }
        long currentTime = System.nanoTime();
        if (currentTime - this.lastPerformanceCheck > 1000000000L) {
            this.performanceCheck();
            this.lastPerformanceCheck = currentTime;
        }
        if (this.emergencyMode) {
            this.handleEmergencyOptimization(server);
        } else {
            this.performStandardOptimization(server);
        }
    }

    private void performanceCheck() {
        long total = this.totalPredictions.get();
        long correct = this.correctPredictions.get();
        if (total > 100L) {
            double accuracy = (double)correct / (double)total;
            double newThreshold = Math.max(0.5, Math.min(0.9, accuracy * 0.9));
            this.adaptiveThreshold.set(newThreshold);
            if (accuracy < 0.6) {
                this.emergencyMode = true;
                SyncFixMod.LOGGER.warn("AI accuracy dropped to {}%, entering emergency mode", (Object)Math.round(accuracy * 100.0));
            } else if (accuracy > 0.8 && this.emergencyMode) {
                this.emergencyMode = false;
                SyncFixMod.LOGGER.info("AI accuracy recovered to {}%, exiting emergency mode", (Object)Math.round(accuracy * 100.0));
            }
        }
        this.entityProfiles.entrySet().removeIf(entry -> ((AIEntityProfile)entry.getValue()).isStale());
    }

    private void handleEmergencyOptimization(MinecraftServer server) {
        for (ServerLevel level : server.m_129785_()) {
            level.m_142646_().m_142273_().forEach(entity -> {
                AIEntityProfile profile;
                if (entity instanceof LivingEntity && !(entity instanceof Player) && (profile = this.entityProfiles.get(entity.m_20148_())) != null) {
                    profile.emergencyOptimize();
                }
            });
        }
    }

    private void performStandardOptimization(MinecraftServer server) {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (ServerLevel level : server.m_129785_()) {
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> this.optimizeWorldEntities(level), this.aiExecutor);
            futures.add(future);
        }
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).orTimeout(25L, TimeUnit.MILLISECONDS).exceptionally(throwable -> null);
    }

    private void optimizeWorldEntities(ServerLevel level) {
        level.m_142646_().m_142273_().forEach(entity -> {
            AIEntityProfile profile;
            if (entity instanceof LivingEntity && !(entity instanceof Player) && (profile = this.entityProfiles.get(entity.m_20148_())) != null) {
                profile.updateLearning((Entity)entity);
            }
        });
    }

    public void shutdown() {
        this.aiExecutor.shutdown();
        try {
            if (!this.aiExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.aiExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.aiExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private int countNearbyPlayers(ServerLevel level, Vec3 position) {
        return (int)level.m_6907_().stream().filter(player -> player.m_20238_(position) <= 16384.0).count();
    }

    private class AIEntityProfile {
        private final UUID entityId;
        private final double[][] neuralWeights;
        private final double[] neuralBiases;
        private final AtomicLong lastUpdate = new AtomicLong(0L);
        private final AtomicInteger predictionCount = new AtomicInteger(0);
        private final AtomicInteger correctCount = new AtomicInteger(0);
        private final Queue<Double> recentInputs = new ConcurrentLinkedQueue<Double>();
        private volatile boolean isLearning = true;
        private volatile double confidence = 0.5;

        public AIEntityProfile(Entity entity) {
            this.entityId = entity.m_20148_();
            this.neuralWeights = this.initializeWeights();
            this.neuralBiases = new double[3];
            Random random = new Random(entity.m_20148_().hashCode());
            for (int i = 0; i < 3; ++i) {
                this.neuralBiases[i] = random.nextGaussian() * 0.1;
            }
        }

        private double[][] initializeWeights() {
            double[][] weights = new double[3][];
            Random random = new Random(this.entityId.hashCode());
            for (int layer = 0; layer < 3; ++layer) {
                int inputSize = layer == 0 ? 8 : 16;
                int outputSize = layer == 2 ? 1 : 16;
                weights[layer] = new double[inputSize * outputSize];
                double stddev = Math.sqrt(2.0 / (double)inputSize);
                for (int i = 0; i < weights[layer].length; ++i) {
                    weights[layer][i] = random.nextGaussian() * stddev;
                }
            }
            return weights;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean predictOptimization(Entity entity, EntityData data) {
            if (AITickOptimizer.this.activeNeuralNetworks.get() > 8) {
                return data != null && data.shouldOptimize();
            }
            AITickOptimizer.this.activeNeuralNetworks.incrementAndGet();
            try {
                boolean shouldOptimize;
                double[] inputs = this.extractFeatures(entity, data);
                double prediction = this.forwardPass(inputs);
                boolean bl = shouldOptimize = prediction > AITickOptimizer.this.adaptiveThreshold.get();
                if (this.isLearning && this.predictionCount.incrementAndGet() % 10 == 0) {
                    this.updateWeights(inputs, shouldOptimize ? 1.0 : 0.0);
                }
                boolean bl2 = shouldOptimize;
                return bl2;
            }
            finally {
                AITickOptimizer.this.activeNeuralNetworks.decrementAndGet();
            }
        }

        private double[] extractFeatures(Entity entity, EntityData data) {
            double[] features = new double[8];
            if (data != null) {
                features[0] = Math.min(1.0, (double)data.getTicksSinceLastMovement() / 100.0);
                features[1] = Math.min(1.0, data.getMovementVariance());
                features[2] = data.isStationary() ? 1.0 : 0.0;
                features[3] = Math.min(1.0, (double)data.getOptimizationLevel() / 5.0);
            }
            ServerLevel level = (ServerLevel)entity.m_9236_();
            int nearbyPlayers = AITickOptimizer.this.countNearbyPlayers(level, entity.m_20182_());
            features[4] = Math.min(1.0, (double)nearbyPlayers / 10.0);
            double distanceToNearestPlayer = level.m_6907_().stream().mapToDouble(p -> p.m_20270_(entity)).min().orElse(1000.0);
            features[5] = Math.min(1.0, distanceToNearestPlayer / 100.0);
            if (entity instanceof Mob) {
                Mob mob = (Mob)entity;
                Brain brain = mob.m_6274_();
                features[6] = brain != null && brain.m_21968_().isPresent() ? 1.0 : 0.0;
                features[7] = mob.m_21525_() ? 1.0 : 0.0;
            }
            return features;
        }

        private double forwardPass(double[] inputs) {
            double[] current = (double[])inputs.clone();
            for (int layer = 0; layer < 3; ++layer) {
                current = this.processLayer(current, layer);
            }
            return this.sigmoid(current[0]);
        }

        private double[] processLayer(double[] inputs, int layer) {
            int inputSize = inputs.length;
            int outputSize = layer == 2 ? 1 : 16;
            double[] outputs = new double[outputSize];
            for (int out = 0; out < outputSize; ++out) {
                double sum = this.neuralBiases[layer];
                for (int in = 0; in < inputSize; ++in) {
                    sum += inputs[in] * this.neuralWeights[layer][in * outputSize + out];
                }
                outputs[out] = layer == 2 ? sum : this.relu(sum);
            }
            return outputs;
        }

        private void updateWeights(double[] inputs, double target) {
            if (!this.isLearning) {
                return;
            }
            double prediction = this.forwardPass(inputs);
            double error = target - prediction;
            if (Math.abs(error) < 0.1) {
                this.correctCount.incrementAndGet();
            }
            this.confidence = Math.min(0.95, this.confidence + (Math.abs(error) < 0.2 ? 0.01 : -0.005));
            if (this.confidence < 0.3) {
                this.reinitializeWeights();
            }
        }

        private void reinitializeWeights() {
            Random random = new Random(System.nanoTime());
            for (int layer = 0; layer < 3; ++layer) {
                for (int i = 0; i < this.neuralWeights[layer].length; ++i) {
                    this.neuralWeights[layer][i] = random.nextGaussian() * 0.1;
                }
            }
            this.confidence = 0.5;
        }

        public void updateLearning(Entity entity) {
            this.lastUpdate.set(System.nanoTime());
        }

        public void emergencyOptimize() {
            this.isLearning = false;
            this.confidence = Math.max(0.1, this.confidence * 0.9);
        }

        public boolean isStale() {
            return System.nanoTime() - this.lastUpdate.get() > 300000000000L;
        }

        private double sigmoid(double x) {
            return 1.0 / (1.0 + Math.exp(-Math.max(-500.0, Math.min(500.0, x))));
        }

        private double relu(double x) {
            return Math.max(0.0, x);
        }
    }
}

