/*
 * Decompiled with CFR 0.152.
 */
package org.syncfix.optimization;

import java.util.Arrays;
import java.util.Queue;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.Deflater;
import net.minecraft.network.protocol.Packet;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.phys.Vec3;
import org.syncfix.SyncFixMod;
import org.syncfix.optimization.EntityData;

public class NetworkOptimizer {
    private static final int PACKET_PREDICTION_WINDOW = 16;
    private static final int MAX_COMPRESSION_LEVEL = 6;
    private static final double PREDICTION_ACCURACY_THRESHOLD = 0.85;
    private static final int ADAPTIVE_SYNC_LEVELS = 8;
    private static final long NETWORK_ANALYSIS_INTERVAL = 5000L;
    private final ConcurrentHashMap<UUID, NetworkProfile> networkProfiles = new ConcurrentHashMap();
    private final ConcurrentHashMap<UUID, PacketPredictor> packetPredictors = new ConcurrentHashMap();
    private final ThreadPoolExecutor networkExecutor = new ThreadPoolExecutor(1, Math.max(2, Runtime.getRuntime().availableProcessors() / 4), 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(2000), r -> {
        Thread t = new Thread(r, "SyncFix-Network-" + System.nanoTime());
        t.setDaemon(true);
        t.setPriority(4);
        return t;
    });
    private final AtomicLong totalPacketsProcessed = new AtomicLong(0L);
    private final AtomicLong packetsOptimized = new AtomicLong(0L);
    private final AtomicReference<Double> compressionRatio = new AtomicReference<Double>(1.0);
    private final AtomicInteger activePredictions = new AtomicInteger(0);
    private volatile boolean adaptiveCompression = true;
    private volatile long lastNetworkAnalysis = 0L;

    public void initialize() {
        this.networkExecutor.prestartAllCoreThreads();
        SyncFixMod.LOGGER.info("Revolutionary Network Optimizer initialized with predictive algorithms");
    }

    public CompletableFuture<Boolean> shouldOptimizeAsync(Entity entity, EntityData data) {
        if (entity == null || this.activePredictions.get() > 32) {
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.supplyAsync(() -> {
            this.activePredictions.incrementAndGet();
            try {
                Boolean bl = this.shouldOptimizeSync(entity, data);
                return bl;
            }
            finally {
                this.activePredictions.decrementAndGet();
            }
        }, this.networkExecutor);
    }

    public void optimizeEntitySync(Entity entity, EntityData data) {
        if (entity == null || entity.m_9236_().f_46443_) {
            return;
        }
        UUID entityId = entity.m_20148_();
        NetworkProfile profile = this.networkProfiles.computeIfAbsent(entityId, k -> new NetworkProfile(entity));
        PacketPredictor predictor = this.packetPredictors.computeIfAbsent(entityId, k -> new PacketPredictor());
        boolean shouldSync = profile.shouldSynchronize(entity, data);
        if (shouldSync) {
            predictor.recordSync(entity.m_20182_(), entity.m_146908_(), entity.m_146909_());
            data.resetSyncCounter();
        }
        profile.updateMetrics(shouldSync);
    }

    private boolean shouldOptimizeSync(Entity entity, EntityData data) {
        NetworkProfile profile = this.networkProfiles.get(entity.m_20148_());
        if (profile == null) {
            return false;
        }
        return profile.calculateOptimizationScore(entity, data) > 0.6;
    }

    public Packet<?> optimizePacket(Packet<?> packet, Entity entity) {
        if (entity == null || packet == null) {
            return packet;
        }
        this.totalPacketsProcessed.incrementAndGet();
        PacketPredictor predictor = this.packetPredictors.get(entity.m_20148_());
        if (predictor != null && predictor.canPredict(packet)) {
            Packet<?> optimized = predictor.optimizePacket(packet, entity);
            if (optimized != packet) {
                this.packetsOptimized.incrementAndGet();
            }
            return optimized;
        }
        return packet;
    }

    public byte[] compressPacketData(byte[] data, ServerPlayer player) {
        if (!this.adaptiveCompression || data.length < 64) {
            return data;
        }
        NetworkProfile profile = this.networkProfiles.get(player.m_20148_());
        int compressionLevel = profile != null ? profile.getOptimalCompressionLevel() : 3;
        try {
            Deflater deflater = new Deflater(compressionLevel);
            deflater.setInput(data);
            deflater.finish();
            byte[] compressed = new byte[data.length];
            int compressedLength = deflater.deflate(compressed);
            deflater.end();
            if ((double)compressedLength < (double)data.length * 0.9) {
                this.updateCompressionMetrics(data.length, compressedLength);
                return Arrays.copyOf(compressed, compressedLength);
            }
        }
        catch (Exception e) {
            SyncFixMod.LOGGER.debug("Compression failed for player {}", (Object)player.m_7755_().getString());
        }
        return data;
    }

    private void updateCompressionMetrics(int original, int compressed) {
        double ratio = (double)compressed / (double)original;
        double currentRatio = this.compressionRatio.get();
        this.compressionRatio.set(currentRatio * 0.95 + ratio * 0.05);
    }

    public void performNetworkAnalysis() {
        long currentTime = System.currentTimeMillis();
        if (currentTime - this.lastNetworkAnalysis < 5000L) {
            return;
        }
        this.networkExecutor.execute(() -> {
            this.analyzeNetworkPerformance();
            this.optimizeNetworkParameters();
            this.cleanupStaleProfiles();
        });
        this.lastNetworkAnalysis = currentTime;
    }

    private void analyzeNetworkPerformance() {
        long total = this.totalPacketsProcessed.get();
        long optimized = this.packetsOptimized.get();
        if (total > 1000L) {
            double optimizationRate = (double)optimized / (double)total;
            double compressionEff = 1.0 - this.compressionRatio.get();
            SyncFixMod.LOGGER.debug("Network performance: {}% packets optimized, {}% compression efficiency", (Object)Math.round(optimizationRate * 100.0), (Object)Math.round(compressionEff * 100.0));
            if (optimizationRate < 0.1) {
                this.adaptiveCompression = false;
            } else if (optimizationRate > 0.3) {
                this.adaptiveCompression = true;
            }
        }
    }

    private void optimizeNetworkParameters() {
        this.networkProfiles.values().parallelStream().forEach(profile -> profile.adaptParameters());
    }

    private void cleanupStaleProfiles() {
        long currentTime = System.nanoTime();
        this.networkProfiles.entrySet().removeIf(entry -> currentTime - ((NetworkProfile)entry.getValue()).getLastActivity() > 300000000000L);
        this.packetPredictors.entrySet().removeIf(entry -> currentTime - ((PacketPredictor)entry.getValue()).getLastPrediction() > 300000000000L);
    }

    public void shutdown() {
        this.networkExecutor.shutdown();
        try {
            if (!this.networkExecutor.awaitTermination(3L, TimeUnit.SECONDS)) {
                this.networkExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.networkExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private static class NetworkProfile {
        private final UUID entityId;
        private final AtomicLong lastActivity = new AtomicLong(System.nanoTime());
        private final AtomicInteger syncCount = new AtomicInteger(0);
        private final AtomicInteger skipCount = new AtomicInteger(0);
        private final AtomicReference<Double> averageDistance = new AtomicReference<Double>(0.0);
        private final AtomicInteger compressionLevel = new AtomicInteger(3);
        private final Queue<Long> syncTimes = new ConcurrentLinkedQueue<Long>();
        private volatile double[] adaptiveThresholds = new double[8];
        private volatile long lastSyncTime = 0L;

        public NetworkProfile(Entity entity) {
            this.entityId = entity.m_20148_();
            Random random = new Random(this.entityId.hashCode());
            for (int i = 0; i < 8; ++i) {
                this.adaptiveThresholds[i] = 0.1 + random.nextDouble() * 0.4;
            }
        }

        public boolean shouldSynchronize(Entity entity, EntityData data) {
            boolean shouldSync;
            this.lastActivity.set(System.nanoTime());
            if (data == null) {
                return true;
            }
            double score = this.calculateOptimizationScore(entity, data);
            int level = Math.min(7, (int)(score * 8.0));
            long currentTime = System.currentTimeMillis();
            long timeSinceLastSync = currentTime - this.lastSyncTime;
            boolean bl = shouldSync = score < this.adaptiveThresholds[level] || timeSinceLastSync > this.getMaxSyncInterval(score);
            if (shouldSync) {
                this.lastSyncTime = currentTime;
                this.syncTimes.offer(currentTime);
                if (this.syncTimes.size() > 20) {
                    this.syncTimes.poll();
                }
            }
            return shouldSync;
        }

        public double calculateOptimizationScore(Entity entity, EntityData data) {
            double score = 0.0;
            if (data != null) {
                score += Math.min(1.0, (double)data.getTicksSinceLastMovement() / 100.0) * 0.3;
                score += data.isStationary() ? 0.4 : 0.0;
                score += Math.min(1.0, data.getMovementVariance()) * 0.2;
            }
            double nearestPlayerDistance = entity.m_9236_().m_6907_().stream().mapToDouble(p -> p.m_20270_(entity)).min().orElse(1000.0);
            return Math.max(0.0, Math.min(1.0, score += Math.min(1.0, nearestPlayerDistance / 200.0) * 0.1));
        }

        private long getMaxSyncInterval(double optimizationScore) {
            return (long)(50.0 + optimizationScore * 950.0);
        }

        public void updateMetrics(boolean synced) {
            if (synced) {
                this.syncCount.incrementAndGet();
            } else {
                this.skipCount.incrementAndGet();
            }
        }

        public void adaptParameters() {
            int skips;
            int syncs = this.syncCount.get();
            if (syncs + (skips = this.skipCount.get()) > 100) {
                double skipRatio = (double)skips / (double)(syncs + skips);
                for (int i = 0; i < 8; ++i) {
                    double adjustment = (skipRatio - 0.7) * 0.01;
                    this.adaptiveThresholds[i] = Math.max(0.05, Math.min(0.95, this.adaptiveThresholds[i] + adjustment));
                }
                if (skipRatio > 0.8) {
                    this.compressionLevel.set(Math.min(6, this.compressionLevel.get() + 1));
                } else if (skipRatio < 0.5) {
                    this.compressionLevel.set(Math.max(1, this.compressionLevel.get() - 1));
                }
            }
        }

        public int getOptimalCompressionLevel() {
            return this.compressionLevel.get();
        }

        public long getLastActivity() {
            return this.lastActivity.get();
        }
    }

    private static class PacketPredictor {
        private final Queue<Vec3> positionHistory = new ConcurrentLinkedQueue<Vec3>();
        private final Queue<Float> rotationHistory = new ConcurrentLinkedQueue<Float>();
        private final AtomicLong lastPrediction = new AtomicLong(System.nanoTime());
        private final AtomicInteger correctPredictions = new AtomicInteger(0);
        private final AtomicInteger totalPredictions = new AtomicInteger(0);
        private volatile Vec3 predictedPosition = Vec3.f_82478_;
        private volatile float predictedRotation = 0.0f;

        private PacketPredictor() {
        }

        public void recordSync(Vec3 position, float yRot, float xRot) {
            this.lastPrediction.set(System.nanoTime());
            this.positionHistory.offer(position);
            this.rotationHistory.offer(Float.valueOf(yRot));
            if (this.positionHistory.size() > 16) {
                this.positionHistory.poll();
            }
            if (this.rotationHistory.size() > 16) {
                this.rotationHistory.poll();
            }
            this.updatePredictions();
        }

        private void updatePredictions() {
            int len;
            if (this.positionHistory.size() >= 3) {
                Vec3[] positions = this.positionHistory.toArray(new Vec3[0]);
                len = positions.length;
                Vec3 velocity = positions[len - 1].m_82546_(positions[len - 2]);
                Vec3 acceleration = len >= 3 ? positions[len - 1].m_82546_(positions[len - 2]).m_82546_(positions[len - 2].m_82546_(positions[len - 3])) : Vec3.f_82478_;
                this.predictedPosition = positions[len - 1].m_82549_(velocity).m_82549_(acceleration.m_82490_(0.5));
            }
            if (this.rotationHistory.size() >= 2) {
                Float[] rotations = this.rotationHistory.toArray(new Float[0]);
                len = rotations.length;
                float rotVelocity = rotations[len - 1].floatValue() - rotations[len - 2].floatValue();
                this.predictedRotation = rotations[len - 1].floatValue() + rotVelocity;
            }
        }

        public boolean canPredict(Packet<?> packet) {
            return this.positionHistory.size() >= 3 && this.getAccuracy() > 0.85;
        }

        public Packet<?> optimizePacket(Packet<?> packet, Entity entity) {
            this.totalPredictions.incrementAndGet();
            Vec3 actualPos = entity.m_20182_();
            double predictionError = this.predictedPosition.m_82554_(actualPos);
            if (predictionError < 0.1) {
                this.correctPredictions.incrementAndGet();
                return null;
            }
            return packet;
        }

        private double getAccuracy() {
            int total = this.totalPredictions.get();
            int correct = this.correctPredictions.get();
            return total > 10 ? (double)correct / (double)total : 0.5;
        }

        public long getLastPrediction() {
            return this.lastPrediction.get();
        }
    }
}

