/*
 * Decompiled with CFR 0.152.
 */
package org.syncfix.optimization;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.ChunkPos;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.server.ServerLifecycleHooks;
import org.syncfix.SyncFixMod;
import org.syncfix.optimization.AITickOptimizer;
import org.syncfix.optimization.EntityData;
import org.syncfix.optimization.EntitySynchronizer;
import org.syncfix.optimization.NetworkOptimizer;

@Mod.EventBusSubscriber(modid="sync_fix")
public class OptimizationManager {
    private static final int REVOLUTIONARY_THREAD_COUNT = Math.max(4, Runtime.getRuntime().availableProcessors());
    private static final int ADAPTIVE_BATCH_SIZE_MIN = 16;
    private static final int ADAPTIVE_BATCH_SIZE_MAX = 512;
    private static final double PERFORMANCE_TARGET_TPS = 19.5;
    private static final long NANO_PER_TICK = 50000000L;
    private static final int PREDICTION_HORIZON = 20;
    private final ForkJoinPool revolutionaryExecutor = new ForkJoinPool(REVOLUTIONARY_THREAD_COUNT, ForkJoinPool.defaultForkJoinWorkerThreadFactory, (t, e) -> SyncFixMod.LOGGER.error("Revolutionary executor error", e), true);
    private final ScheduledExecutorService adaptiveScheduler = Executors.newScheduledThreadPool(2, r -> {
        Thread t = new Thread(r, "SyncFix-Adaptive-" + System.nanoTime());
        t.setDaemon(true);
        t.setPriority(9);
        return t;
    });
    private final ConcurrentHashMap<UUID, EntityData> entityProfiles = new ConcurrentHashMap();
    private final ConcurrentHashMap<ChunkPos, ChunkOptimizationProfile> chunkProfiles = new ConcurrentHashMap();
    private final PriorityBlockingQueue<OptimizationTask> taskQueue = new PriorityBlockingQueue(1000);
    private final AtomicLong totalOptimizations = new AtomicLong(0L);
    private final AtomicReference<Double> averageTickTime = new AtomicReference<Double>(5.0E7);
    private final AtomicInteger adaptiveBatchSize = new AtomicInteger(64);
    private final AtomicBoolean emergencyMode = new AtomicBoolean(false);
    private final AtomicLong lastPerformanceAnalysis = new AtomicLong(0L);
    private final NetworkOptimizer networkOptimizer = new NetworkOptimizer();
    private final AITickOptimizer aiOptimizer = new AITickOptimizer();
    private final EntitySynchronizer entitySynchronizer = new EntitySynchronizer();
    private volatile boolean initialized = false;
    private volatile PerformancePredictor performancePredictor;
    private volatile SystemResourceMonitor resourceMonitor;
    private volatile AdaptiveLoadBalancer loadBalancer;

    public void initialize() {
        if (this.initialized) {
            return;
        }
        CompletableFuture.runAsync(() -> {
            try {
                this.performancePredictor = new PerformancePredictor();
                this.resourceMonitor = new SystemResourceMonitor();
                this.loadBalancer = new AdaptiveLoadBalancer();
                this.networkOptimizer.initialize();
                this.aiOptimizer.initialize();
                this.entitySynchronizer.initialize();
                this.startAdaptiveOptimization();
                this.initialized = true;
                SyncFixMod.LOGGER.info("Revolutionary Optimization Manager initialized with {} threads", (Object)REVOLUTIONARY_THREAD_COUNT);
            }
            catch (Exception e) {
                SyncFixMod.LOGGER.error("Failed to initialize optimization manager", (Throwable)e);
            }
        }, this.revolutionaryExecutor);
    }

    private void startAdaptiveOptimization() {
        this.adaptiveScheduler.scheduleAtFixedRate(this::performSystemAnalysis, 0L, 100L, TimeUnit.MILLISECONDS);
        this.adaptiveScheduler.scheduleAtFixedRate(this::adaptiveParameterTuning, 0L, 1L, TimeUnit.SECONDS);
        this.adaptiveScheduler.scheduleAtFixedRate(this::predictiveOptimization, 0L, 5L, TimeUnit.SECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @SubscribeEvent
    public void onServerTick(TickEvent.ServerTickEvent event) {
        if (event.phase != TickEvent.Phase.END || !this.initialized) {
            return;
        }
        long tickStart = System.nanoTime();
        MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
        if (server == null) {
            return;
        }
        try {
            if (this.emergencyMode.get()) {
                this.performEmergencyOptimization(server);
            } else {
                this.performRevolutionaryOptimization(server);
            }
        }
        catch (Exception e) {
            SyncFixMod.LOGGER.error("Critical error in optimization cycle", (Throwable)e);
            this.emergencyMode.set(true);
        }
        finally {
            long tickDuration = System.nanoTime() - tickStart;
            this.updatePerformanceMetrics(tickDuration);
        }
    }

    private void performRevolutionaryOptimization(MinecraftServer server) {
        ArrayList<CompletableFuture<Void>> optimizationFutures = new ArrayList<CompletableFuture<Void>>();
        for (ServerLevel level : server.m_129785_()) {
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> this.optimizeWorldRevolutionary(level), this.revolutionaryExecutor).orTimeout(40L, TimeUnit.MILLISECONDS);
            optimizationFutures.add(future);
        }
        CompletableFuture.allOf(optimizationFutures.toArray(new CompletableFuture[0])).exceptionally(throwable -> {
            SyncFixMod.LOGGER.warn("Some optimization tasks timed out or failed");
            return null;
        });
    }

    private void optimizeWorldRevolutionary(ServerLevel level) {
        ArrayList<Entity> entities = new ArrayList<Entity>();
        level.m_142646_().m_142273_().forEach(entities::add);
        int batchSize = this.adaptiveBatchSize.get();
        List<List<Entity>> batches = this.partitionEntities(entities, batchSize);
        ArrayList<CompletableFuture<Void>> batchFutures = new ArrayList<CompletableFuture<Void>>();
        for (List<Entity> batch : batches) {
            CompletableFuture<Void> batchFuture = CompletableFuture.runAsync(() -> this.processBatchRevolutionary(batch, level), this.revolutionaryExecutor);
            batchFutures.add(batchFuture);
        }
        CompletableFuture.allOf(batchFutures.toArray(new CompletableFuture[0])).orTimeout(35L, TimeUnit.MILLISECONDS).exceptionally(throwable -> null);
    }

    private void processBatchRevolutionary(List<Entity> entities, ServerLevel level) {
        for (Entity entity : entities) {
            try {
                EntityData data = this.entityProfiles.computeIfAbsent(entity.m_20148_(), k -> new EntityData());
                data.update(entity);
                if (data.shouldOptimize()) {
                    CompletableFuture<Boolean> aiDecision = this.aiOptimizer.shouldSkipTickAsync(entity, data);
                    CompletableFuture<Void> syncOptimization = this.entitySynchronizer.optimizeEntityBehaviorAsync(entity, data);
                    CompletableFuture<Boolean> networkDecision = this.networkOptimizer.shouldOptimizeAsync(entity, data);
                    ((CompletableFuture)CompletableFuture.allOf(aiDecision, syncOptimization, networkDecision).orTimeout(10L, TimeUnit.MILLISECONDS).thenRun(() -> {
                        this.networkOptimizer.optimizeEntitySync(entity, data);
                        this.totalOptimizations.incrementAndGet();
                    })).exceptionally(throwable -> null);
                } else {
                    this.networkOptimizer.optimizeEntitySync(entity, data);
                }
                this.updateChunkProfile(entity, level);
            }
            catch (Exception e) {
                SyncFixMod.LOGGER.debug("Entity optimization error: {}", (Object)e.getMessage());
            }
        }
    }

    private void updateChunkProfile(Entity entity, ServerLevel level) {
        ChunkPos chunkPos = new ChunkPos(entity.m_20183_());
        ChunkOptimizationProfile profile = this.chunkProfiles.computeIfAbsent(chunkPos, k -> new ChunkOptimizationProfile());
        profile.recordEntityActivity(entity);
    }

    private List<List<Entity>> partitionEntities(Collection<Entity> entities, int batchSize) {
        ArrayList<List<Entity>> batches = new ArrayList<List<Entity>>();
        ArrayList<Entity> currentBatch = new ArrayList<Entity>();
        for (Entity entity : entities) {
            currentBatch.add(entity);
            if (currentBatch.size() < batchSize) continue;
            batches.add(new ArrayList(currentBatch));
            currentBatch.clear();
        }
        if (!currentBatch.isEmpty()) {
            batches.add(currentBatch);
        }
        return batches;
    }

    private void performEmergencyOptimization(MinecraftServer server) {
        SyncFixMod.LOGGER.warn("Emergency optimization mode active");
        for (ServerLevel level : server.m_129785_()) {
            level.m_142646_().m_142273_().forEach(entity -> {
                EntityData data = this.entityProfiles.get(entity.m_20148_());
                if (data != null && data.shouldOptimize()) {
                    this.networkOptimizer.optimizeEntitySync((Entity)entity, data);
                }
            });
        }
        if (this.averageTickTime.get() < 6.0E7) {
            this.emergencyMode.set(false);
            SyncFixMod.LOGGER.info("Exiting emergency optimization mode");
        }
    }

    private void updatePerformanceMetrics(long tickDuration) {
        double currentAvg = this.averageTickTime.get();
        double newAvg = currentAvg * 0.95 + (double)tickDuration * 0.05;
        this.averageTickTime.set(newAvg);
        if ((double)tickDuration > 1.0E8) {
            this.emergencyMode.set(true);
        }
    }

    private void performSystemAnalysis() {
        if (!this.initialized) {
            return;
        }
        long currentTime = System.nanoTime();
        if (currentTime - this.lastPerformanceAnalysis.get() < 1000000000L) {
            return;
        }
        try {
            this.resourceMonitor.updateMetrics();
            this.performancePredictor.updatePredictions();
            double memoryPressure = this.resourceMonitor.getMemoryPressure();
            double cpuLoad = this.resourceMonitor.getCpuLoad();
            if (memoryPressure > 0.8 || cpuLoad > 0.9) {
                this.emergencyMode.set(true);
            } else if (memoryPressure < 0.6 && cpuLoad < 0.7) {
                this.emergencyMode.set(false);
            }
            this.lastPerformanceAnalysis.set(currentTime);
        }
        catch (Exception e) {
            SyncFixMod.LOGGER.debug("System analysis error: {}", (Object)e.getMessage());
        }
    }

    private void adaptiveParameterTuning() {
        if (!this.initialized) {
            return;
        }
        double avgTick = this.averageTickTime.get();
        double targetRatio = avgTick / 5.0E7;
        int currentBatch = this.adaptiveBatchSize.get();
        int newBatch = targetRatio > 1.1 ? Math.max(16, (int)((double)currentBatch * 0.9)) : (targetRatio < 0.8 ? Math.min(512, (int)((double)currentBatch * 1.05)) : currentBatch);
        this.adaptiveBatchSize.set(newBatch);
        this.entityProfiles.entrySet().removeIf(entry -> ((EntityData)entry.getValue()).isStale());
        this.chunkProfiles.entrySet().removeIf(entry -> ((ChunkOptimizationProfile)entry.getValue()).isInactive());
    }

    private void predictiveOptimization() {
        if (!this.initialized || this.performancePredictor == null) {
            return;
        }
        try {
            double predictedLoad = this.performancePredictor.predictNextTickLoad();
            if (predictedLoad > 1.5) {
                this.adaptiveBatchSize.set(Math.max(16, this.adaptiveBatchSize.get() / 2));
                SyncFixMod.LOGGER.debug("Predictive optimization: reducing batch size due to predicted load: {}", (Object)predictedLoad);
            }
        }
        catch (Exception e) {
            SyncFixMod.LOGGER.debug("Predictive optimization error: {}", (Object)e.getMessage());
        }
    }

    public void shutdown() {
        this.revolutionaryExecutor.shutdown();
        this.adaptiveScheduler.shutdown();
        try {
            if (!this.revolutionaryExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.revolutionaryExecutor.shutdownNow();
            }
            if (!this.adaptiveScheduler.awaitTermination(3L, TimeUnit.SECONDS)) {
                this.adaptiveScheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.revolutionaryExecutor.shutdownNow();
            this.adaptiveScheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        if (this.networkOptimizer != null) {
            this.networkOptimizer.shutdown();
        }
        if (this.aiOptimizer != null) {
            this.aiOptimizer.shutdown();
        }
        if (this.entitySynchronizer != null) {
            this.entitySynchronizer.shutdown();
        }
    }

    private static class ChunkOptimizationProfile {
        private final AtomicInteger entityCount = new AtomicInteger(0);
        private final AtomicLong lastActivity = new AtomicLong(System.nanoTime());
        private final AtomicReference<Double> optimizationScore = new AtomicReference<Double>(0.5);

        private ChunkOptimizationProfile() {
        }

        void recordEntityActivity(Entity entity) {
            this.lastActivity.set(System.nanoTime());
            this.entityCount.incrementAndGet();
        }

        boolean isInactive() {
            return System.nanoTime() - this.lastActivity.get() > 300000000000L;
        }
    }

    private static class SystemResourceMonitor {
        private final MemoryMXBean memoryBean = ManagementFactory.getMemoryMXBean();
        private final ThreadMXBean threadBean = ManagementFactory.getThreadMXBean();
        private final AtomicReference<Double> memoryPressure = new AtomicReference<Double>(0.0);
        private final AtomicReference<Double> cpuLoad = new AtomicReference<Double>(0.0);

        private SystemResourceMonitor() {
        }

        void updateMetrics() {
            long usedMemory = this.memoryBean.getHeapMemoryUsage().getUsed();
            long maxMemory = this.memoryBean.getHeapMemoryUsage().getMax();
            this.memoryPressure.set((double)usedMemory / (double)maxMemory);
            int threadCount = this.threadBean.getThreadCount();
            this.cpuLoad.set(Math.min(1.0, (double)threadCount / (double)REVOLUTIONARY_THREAD_COUNT));
        }

        double getMemoryPressure() {
            return this.memoryPressure.get();
        }

        double getCpuLoad() {
            return this.cpuLoad.get();
        }
    }

    private static class PerformancePredictor {
        private final Queue<Double> tickTimeHistory = new ConcurrentLinkedQueue<Double>();
        private final AtomicReference<Double> predictedLoad = new AtomicReference<Double>(1.0);

        private PerformancePredictor() {
        }

        void updatePredictions() {
            if (this.tickTimeHistory.size() > 20) {
                this.tickTimeHistory.poll();
            }
        }

        double predictNextTickLoad() {
            return this.predictedLoad.get();
        }
    }

    private static class AdaptiveLoadBalancer {
        private final AtomicInteger currentStrategy = new AtomicInteger(0);

        private AdaptiveLoadBalancer() {
        }

        void balanceLoad() {
        }
    }

    private static class OptimizationTask
    implements Comparable<OptimizationTask> {
        final Runnable task;
        final int priority;
        final long timestamp;

        OptimizationTask(Runnable task, int priority) {
            this.task = task;
            this.priority = priority;
            this.timestamp = System.nanoTime();
        }

        @Override
        public int compareTo(OptimizationTask other) {
            int priorityCompare = Integer.compare(other.priority, this.priority);
            return priorityCompare != 0 ? priorityCompare : Long.compare(this.timestamp, other.timestamp);
        }
    }
}

